
from webapp import *
import calendar

MODE=["VIREMENT", "MVOLA", "ESPECE", "CHEQUE"]

class Fournisseur(db.Model):
    __tablename__='_fournisseur'
    id=db.Column(db.Integer, primary_key=True)
    code=db.Column(db.String(30), default="")
    nom=db.Column(db.String(256), default="")
    credit=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    debit=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    credit_bilan=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    debit_bilan=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    @property
    def solde(self):
        return self.solde_bilan + self.debit - self.credit
    @property
    def solde_bilan(self):
        return self.debit_bilan - self.credit_bilan

class CommandeFournisseur(db.Model):
    __tablename__="_commande_fournisseur"
    id=db.Column(db.Integer, primary_key=True)
    numero=db.Column(db.String(50)) # 1/F/20  1/F/21
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    fournisseur_id=db.Column(db.Integer, db.ForeignKey('_fournisseur.id'))
    fournisseur=relationship("Fournisseur", backref='commande_fournisseur')
    montant_commande=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    # status_reg=db.Column(db.String(20), default="NON REGLE") #REGLE , PARTIEL
    # status_livraison=db.Column(db.String(20), default="NON LIVRE") #PARTIEL , LIVRE
    
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")
    @property
    def date_html(self):
        return self.date.strftime("%Y-%m-%d")
    @property
    def montant_str(self):
        return str(self.montant_commande)

    def statut_livraison(self):
        detail_com=self.detail_commande_fournisseur
        livraison=self.livraison_fournisseur
        LIVRE=0
        COMMANDE=0
        if not livraison:
            return "0%"
        else:
            detail_liv=[]
            for l in livraison:
                detail_liv.extend(l.livraison_fournisseur_detail)
            for liv in detail_liv:
                LIVRE+=liv.quantite
            for com in detail_com:
                COMMANDE+=com.quantite
            return "{}%".format(int(LIVRE/COMMANDE*100))
    def statut_reglement(self):
        reglements=self.versement_fournisseur
        total=Decimal("0")
        for reg in reglements:
            total+=reg.montant
        reste=self.montant_commande-total

        return total , "NON REGLEE" if self.montant_commande>total else "REGLEE", reste





class DetailCommandeFournisseur(db.Model):
    __tablename__="_detail_commande_fournisseur"
    id=db.Column(db.Integer, primary_key=True)
    commande_fournisseur_id=db.Column(db.Integer, db.ForeignKey('_commande_fournisseur.id'))
    commande_fournisseur=relationship("CommandeFournisseur", backref='detail_commande_fournisseur')
    article_id=db.Column(db.Integer, db.ForeignKey('_article.id'))
    article=relationship("Article", backref='detail_commande_fournisseur')
    article_code=db.Column(db.String(30), default="")
    article_name=db.Column(db.String(256), default="")
    prix_achat=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    quantite=db.Column(db.Integer, default=0)
    @property
    def prix_achat_str(self):
        return str(self.prix_achat)
    @property
    def sous_total(self):
        return self.prix_achat*Decimal(self.quantite)

    @property
    def sous_total_str(self):
        return str(self.sous_total)



class BalanceFournisseur(db.Model):
    __tablename__="_balance_fournisseur"
    id=db.Column(db.Integer, primary_key=True)
    affectation=db.Column(db.String(100), default="")
    date = db.Column(db.Date, default=datetime.now().date())
    fournisseur_id = db.Column(db.Integer, db.ForeignKey('_fournisseur.id'))
    fournisseur = relationship('Fournisseur', backref='balance_fournisseur')
    credit = db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0)) #livraison
    debit= db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0)) #paiement 
    solde=db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0))
    versement_fournisseur_id=db.Column(db.Integer, db.ForeignKey('_versement_fournisseur.id'))
    versement_fournisseur = relationship('VersementFournisseur', backref='balance_fournisseur')
    commande_fournisseur_id=db.Column(db.Integer, db.ForeignKey('_commande_fournisseur.id'))
    commande_fournisseur = relationship('CommandeFournisseur', backref='balance_fournisseur')

    @property
    def ref(self):
        if self.versement_fournisseur_id:
            return self.versement_fournisseur.numero
        elif self.commande_fournisseur_id:
            return self.commande_fournisseur.numero
        else:
            return ""
    @property
    def date_html(self):
        return self.date.strftime("%Y-%m-%d")
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")


class VersementFournisseur(db.Model):
    __tablename__="_versement_fournisseur"
    id=db.Column(db.Integer, primary_key=True)
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    numero=db.Column(db.String(50), default="") #num depense
    fournisseur_id=db.Column(db.Integer, db.ForeignKey('_fournisseur.id'))
    fournisseur=relationship("Fournisseur", backref='versement_fournisseur')
    commande_fournisseur_id=db.Column(db.Integer, db.ForeignKey('_commande_fournisseur.id'))
    commande_fournisseur=relationship("CommandeFournisseur", backref='versement_fournisseur')
    sortie_caisse_id=db.Column(db.Integer, db.ForeignKey('_sortie_caisse.id'))
    sortie_caisse=relationship("SortieCaisse", backref='versement_fournisseur')
    montant=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    frais=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    mode=db.Column(db.String(50), default="")
    ref=db.Column(db.String(256), default="")
    piece_joint=db.Column(db.String(256), default="none.jpg")
    statut_encaissement = db.Column(db.String(20), default='NON ENCAISSE')  #'non encaissé', 'encaissé', 'annulé',
    @property
    def date_html(self):
        return self.date.strftime("%Y-%m-%d")
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")

    @property
    def montant_str(self):
        return str(self.montant)





class LivraisonFournisseur(db.Model):
    __tablename__="_livraison_fournisseur"
    id=db.Column(db.Integer, primary_key=True)
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    fournisseur_id=db.Column(db.Integer, db.ForeignKey('_fournisseur.id'))
    fournisseur=relationship("Fournisseur", backref='livraison_fournisseur')
    commande_fournisseur_id=db.Column(db.Integer, db.ForeignKey('_commande_fournisseur.id'))
    commande_fournisseur=relationship("CommandeFournisseur", backref='livraison_fournisseur')
    ref=db.Column(db.String(100), default="")
    @property
    def date_html(self):
        return self.date.strftime("%Y-%m-%d")
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")



class LivraisonFournisseurDetail(db.Model):
    __tablename__="_livraison_fournisseur_detail"
    id=db.Column(db.Integer, primary_key=True)
    livraison_fournisseur_id=db.Column(db.Integer, db.ForeignKey('_livraison_fournisseur.id'))
    livraison_fournisseur=relationship("LivraisonFournisseur", backref='livraison_fournisseur_detail')
    article_id=db.Column(db.Integer, db.ForeignKey('_article.id'))
    article=relationship("Article", backref='livraison_fournisseur_detail')
    article_code=db.Column(db.String(30), default="")
    article_name=db.Column(db.String(256), default="")
    prix_achat_unit=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    quantite=db.Column(db.Integer, default=0)
    @property
    def prix_achat_str(self):
        return str(self.prix_achat_unit)
    @property
    def sous_total(self):
        return Decimal(self.quantite)*self.prix_achat_unit

    @property
    def sous_total_str(self):
        return str(self.sous_total)
    





class Client(db.Model):
    __tablename__='_client'
    id=db.Column(db.Integer, primary_key=True)
    code=db.Column(db.String(30), default="")
    nom=db.Column(db.String(256), default="")
    phone=db.Column(db.String(50), default="")
    address=db.Column(db.String(100), default="")
    credit=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    debit=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    credit_bilan=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    debit_bilan=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    @property
    def solde(self):
        return self.solde_bilan + self.debit - self.credit
    @property
    def solde_bilan(self):
        return self.debit_bilan - self.credit_bilan

    @property
    def full_name(self):
        return self.nom+"_"+self.phone

class CommandeClient(db.Model):
    __tablename__="_commande_client"
    id=db.Column(db.Integer, primary_key=True)
    numero=db.Column(db.String(50)) # 1/F/20  1/F/21
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    client_id=db.Column(db.Integer, db.ForeignKey('_client.id'))
    client=relationship("Client", backref='commande_client')
    user_id=db.Column(db.Integer, db.ForeignKey('_user.id'))
    user=relationship("User", backref='commande_client')
    net_payer=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    total_remise=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    montant_achat=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    ref=db.Column(db.String(128), default="")
    type_commande=db.Column(db.String(20),default="PROVANDE") #POUSSIN
    status=db.Column(db.String(20),default="ENCOURS") #TERMINE, ANNULE
    
    @property
    def date_html(self):
        return self.date.strftime("%Y-%m-%d")
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")
    @property
    def net_payer_str(self):
        return str(self.net_payer)
    @property
    def total_remise_str(self):
        return str(self.total_remise)
    
    @property
    def marge(self):
        return self.net_payer-self.montant_achat

    @property
    def montant_paye(self):
        all_reg=Reglement.query.filter_by(commande_client=self).all()
        montant=Decimal(0)
        for reg in all_reg:
            montant+=reg.montant
        reste=self.net_payer-montant
        return montant, str(montant), "REGLEE" if self.net_payer<=montant else "NON REGLEE",reste

    def taux_livraison(self):
        all_detail=self.detail_commande_client
        all_livraison=Livraison.query.filter_by(commande_client=self).all()
        qte_comm=0
        qte_livre=0
        if not all_livraison:
            return "NON LIVRE"
        else:
            for detail in all_detail:
                qte_comm+=detail.quantite
            for livraison in all_livraison:
                qte_livre+=livraison.quantite
            percent=(qte_livre/qte_comm)*100
            percent=int(percent)
            return "{}% LIVRE".format(str(percent))


    def date_livraison(self):
        total_liv=set()
        for liv in self.livraison:
            total_liv.add(liv.date)
        return list(total_liv)

    @classmethod
    def current_month(cls):
        now=datetime.now()
        delta_min=datetime(now.year, now.month, 1,0,0,0)
        delta_max=datetime(now.year,now.month,calendar.monthrange(now.year,now.month)[1],23,59,59)
        all_commande=cls.query.filter(cls.date>=delta_min,cls.date<=delta_max).all()
        CUMUL=Decimal(0)
        MARGE=Decimal(0)
        for commande in all_commande:
            CUMUL+=commande.net_payer
            MARGE+=commande.marge
        return CUMUL, MARGE

    @classmethod
    def current_day(cls):
        now=datetime.now()
        delta_min=datetime(now.year, now.month, now.day,0,0,0)
        delta_max=datetime(now.year,now.month,now.day,23,59,59)
        all_commande=cls.query.filter(cls.date>=delta_min,cls.date<=delta_max).all()
        CUMUL=Decimal(0)
        MARGE=Decimal(0)
        for commande in all_commande:
            CUMUL+=commande.net_payer
            MARGE+=commande.marge
        return CUMUL, MARGE



class DetailCommandeClient(db.Model):
    __tablename__="_detail_commande_client"
    id=db.Column(db.Integer, primary_key=True)
    commande_client_id=db.Column(db.Integer, db.ForeignKey('_commande_client.id'))
    commande_client=relationship("CommandeClient", backref='detail_commande_client')
    article_id=db.Column(db.Integer, db.ForeignKey('_article.id'))
    article=relationship("Article", backref='detail_commande_client')
    article_code=db.Column(db.String(30), default="")
    article_name=db.Column(db.String(256), default="")
    prix_unit=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    prix_achat=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    remise=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    quantite=db.Column(db.Integer, default=0)

    @property
    def pu_html(self):
        return str(self.prix_unit)
    @property
    def remise_html(self):
        return str(self.remise)
    @property
    def sous_total(self):
        remise=self.remise*Decimal(self.quantite)
        montant=self.prix_unit*Decimal(self.quantite)
        return montant-remise

    def qty_livre(self):
        qty=0
        livraison=Livraison.query.filter_by(commande_client_id=self.commande_client_id,article_id=self.article_id).all()
        for liv in livraison:
            qty+=liv.quantite
        return qty




class ReseravationPoussin(db.Model):
    __tablename__="_reservation_poussin"
    id=db.Column(db.Integer, primary_key=True)
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    client_id=db.Column(db.Integer, db.ForeignKey('_client.id'))
    client=relationship("Client", backref='reservation_poussin')
    date_reservation=db.Column(db.DateTime, default=db.func.current_timestamp())
    prix=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    quantite_commande=db.Column(db.Integer, default=0)
    quantite_livre=db.Column(db.Integer, default=0)
    status=db.Column(db.String(30), default="NON LIVRE") #LIVRE
    avance=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    commande_client_id=db.Column(db.Integer, db.ForeignKey('_commande_client.id'))
    commande_client=relationship("CommandeClient", backref="reservation_poussin")

    @property
    def date_reservation_str(self):
        return self.date_reservation.strftime("%d/%m/%Y") if self.date_reservation else "-"

    @property
    def prix_str(self):
        return str(self.prix)




class Livraison(db.Model):
    __tablename__="_livraison"
    id=db.Column(db.Integer, primary_key=True)
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    commande_client_id=db.Column(db.Integer, db.ForeignKey('_commande_client.id'))
    commande_client=relationship("CommandeClient", backref='livraison')
    article_id=db.Column(db.Integer, db.ForeignKey('_article.id'))
    article=relationship("Article", backref='livraison')
    quantite=db.Column(db.Integer, default=0)
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")




class Reglement(db.Model):
    __tablename__='_reglement'
    id=db.Column(db.Integer, primary_key=True)
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    commande_client_id=db.Column(db.Integer, db.ForeignKey('_commande_client.id'))
    commande_client=relationship("CommandeClient", backref='reglement')
    montant=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    mode=db.Column(db.String(50), default="")
    numero_recu=db.Column(db.String(50))
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")
    @property
    def montant_str(self):
        return str(self.montant)

class Remboursement(db.Model):
    __tablename__='_remboursement'
    id=db.Column(db.Integer, primary_key=True)
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    commande_client_id=db.Column(db.Integer, db.ForeignKey('_commande_client.id'))
    commande_client=relationship("CommandeClient", backref='remboursement')
    montant=db.Column(db.Numeric(precision=20, scale=2),default=Decimal(0))
    mode=db.Column(db.String(50), default="")
    numero_recu=db.Column(db.String(50))
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")
    @property
    def montant_str(self):
        return str(self.montant)


class Caisse(db.Model):
    """ 3type de caisse, ZO, Mvola, BANQUE"""
    __tablename__="_caisse"
    id=db.Column(db.Integer, primary_key=True)
    name=db.Column(db.String(100))
    mode=db.Column(db.String(100))
    solde = db.Column(db.Numeric(precision=20, scale=2), default=Decimal(0))
    recette_val = db.Column(db.Numeric(precision=20, scale=2), default=Decimal(0))
    depense_val = db.Column(db.Numeric(precision=20, scale=2), default=Decimal(0))
    solde_bilan = db.Column(db.Numeric(precision=20, scale=2), default=Decimal(0))
    user_id = db.Column(db.Integer, db.ForeignKey('_user.id'))
    user = relationship('User', back_populates='caisse')
    principal=db.Column(db.Boolean, default=False)
    @property
    def solde_val(self):
        return self.solde_bilan+self.recette_val-self.depense_val
    @property
    def solde_val_html(self):
        return str(self.solde_val)
    @property
    def solde_html(self):
        return str(self.solde)
    @property
    def recette_html(self):
        return str(self.recette_val)
    @property
    def depense_html(self):
        return str(self.depense_val)

    @classmethod
    def solde_caisses(cls):
        all_caisse=cls.query.all()
        CUMUL=Decimal(0)
        for caisse in all_caisse:
            CUMUL+=caisse.solde_val
        return CUMUL


class SortieCaisse(db.Model):
    __tablename__ = '_sortie_caisse'
    id = db.Column(db.Integer, primary_key=True)
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    numero_fiche = db.Column(db.String(20), nullable=False, unique=True)
    date_sortie = db.Column(db.Date, default=datetime.utcnow)
    caisse_id = db.Column(db.Integer, db.ForeignKey('_caisse.id'))
    caisse = relationship('Caisse', backref='sortie_caisse')
    beneficiaire = db.Column(db.String(100), default="")
    montant = db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0))
    libelle = db.Column(db.String(50), default="")
    objet = db.Column(db.String(255), default="")
    pieces_jointes = db.Column(db.String(255), nullable=True)  # facultatif, pour lister les documents joints
    responsable_caisse = db.Column(db.String(256), nullable=True)
    coursier_signature = db.Column(db.Boolean, default=False)
    responsable_signature = db.Column(db.Boolean, default=False)
    
    def __repr__(self):
        return f"<SortieCaisse {self.numero_fiche}>"
    @property
    def date_str(self):
        return self.date_sortie.strftime("%d/%m/%Y")   

    


class Recette(db.Model):
    __tablename__='_recette'
    id=db.Column(db.Integer, primary_key=True)
    date_creation=db.Column(db.DateTime, default=db.func.current_timestamp())
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    ref=db.Column(db.String(50)) #numero recu paiement
    libelle = db.Column(db.String(256)) #règlement
    affectation = db.Column(db.String(100))#commande numero
    montant = db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0))
    caisse_id = db.Column(db.Integer, db.ForeignKey('_caisse.id'))
    caisse = relationship('Caisse', backref='recette')
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")

class Transfert(db.Model):
    __tablename__="_transfert"
    id=db.Column(db.Integer, primary_key=True)
    date = db.Column(db.DateTime, default=db.func.current_timestamp())
    t_type=db.Column(db.String(20)) #IN OUT
    libelle = db.Column(db.String(256))
    montant= db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0))
    caisse_id = db.Column(db.Integer, db.ForeignKey('_caisse.id'))
    caisse = relationship('Caisse', backref='transfert')
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")
    
    @classmethod
    def make_transfert(cls, date, caisse_in, caisse_out, montant):
        caisse_in=Caisse.query.filter_by(id=caisse_in).first()
        caisse_out=Caisse.query.filter_by(id=caisse_out).first()
        new_in= cls(caisse_id=caisse_in.id, montant=montant, libelle="TRANSFERT",t_type="IN")
        new_out= cls(caisse_id=caisse_out.id, montant=montant, libelle="TRANSFERT",t_type="OUT")
        db.session.add(new_in)
        db.session.add(new_out)
        caisse_in.recette_val+=montant
        caisse_out.depense_val+=montant
        new_depense=SortieCaisse(date_sortie=date,
                        libelle="TRANSFERT OUT",
                        objet="TRANSFERT VERS {}".format(caisse_in.name),
                        caisse=caisse_out,
                        montant=montant,
                        numero_fiche=NumDepense.get_number()
                    )
        new_recette=Recette(date=date,
                        ref="",
                        libelle="TRANSFERT IN",
                        affectation="TRANSFERT VENANT DE {}".format(caisse_out.name),
                        caisse=caisse_in,
                        montant=montant
                    )
        db.session.add(new_recette)
        db.session.add(new_depense)
        new_journal_in=JournalCaisse(
                                date=date,
                                caisse=caisse_in,
                                affectation="TRANSFERT IN",
                                recette=new_recette,
                                entree=montant,
                                solde=caisse_in.solde_val
                            )
        new_journal_out=JournalCaisse(
                                date=date,
                                caisse=caisse_out,
                                affectation="TRANSFERT OUT",
                                sortie_caisse=new_depense,
                                sortie=montant,
                                solde=caisse_out.solde_val
                            )
        db.session.add(new_journal_in)
        db.session.add(new_journal_out)
        db.session.commit()


class JournalCaisse(db.Model):
    __tablename__="_journal_caisse"
    id=db.Column(db.Integer, primary_key=True)
    caisse_id = db.Column(db.Integer, db.ForeignKey('_caisse.id'))
    caisse = relationship('Caisse', backref='journal_caisse')
    date=db.Column(db.DateTime, default=db.func.current_timestamp())
    affectation=db.Column(db.String(100), default="")
    recette_id=db.Column(db.Integer, db.ForeignKey('_recette.id'))
    recette=relationship("Recette", backref="journal_caisse")
    sortie_caisse_id=db.Column(db.Integer, db.ForeignKey('_sortie_caisse.id'))
    sortie_caisse=relationship("SortieCaisse", backref="journal_caisse")
    entree=db.Column(db.Numeric(precision=20, scale=2), default=Decimal(0))
    sortie=db.Column(db.Numeric(precision=20, scale=2), default=Decimal(0))
    solde=db.Column(db.Numeric(precision=20, scale=2), default=Decimal(0))

    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")
    @property
    def ref(self):
        if self.recette_id:
            return self.recette.ref
        elif self.sortie_caisse_id:
            return self.sortie_caisse.numero_fiche
        else:
            return ""



class BalanceTiers(db.Model):
    __tablename__="_balance_tiers"
    id=db.Column(db.Integer, primary_key=True)
    affectation=db.Column(db.String(100), default="")
    date = db.Column(db.Date, default=datetime.now().date())
    client_id = db.Column(db.Integer, db.ForeignKey('_client.id'))
    client = relationship('Client', backref='balance_tiers')
    credit = db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0)) #facture commande
    debit= db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0)) #paiement 
    solde=db.Column(db.Numeric(precision=10, scale=2), default=Decimal(0))
    reglement_id=db.Column(db.Integer, db.ForeignKey('_reglement.id'))
    reglement = relationship('Reglement', backref='balance_tiers')
    remboursement_id=db.Column(db.Integer, db.ForeignKey('_remboursement.id'))
    remboursement = relationship('Remboursement', backref='balance_tiers')
    commande_client_id=db.Column(db.Integer, db.ForeignKey('_commande_client.id'))
    commande_client = relationship('CommandeClient', backref='balance_tiers')

    @property
    def ref(self):
        if self.reglement_id:
            return self.reglement.numero_recu
        elif self.commande_client_id:
            return self.commande_client.numero
        elif self.remboursement_id:
            return self.remboursement.commande_client.numero
        else:
            return ""
    @property
    def date_html(self):
        return self.date.strftime("%Y-%m-%d")
    @property
    def date_str(self):
        return self.date.strftime("%d/%m/%Y")



class NumFacture(db.Model):
    __tablename__="_num_facture"
    id=db.Column(db.Integer, primary_key=True)
    last_number=db.Column(db.String(50))
    @classmethod
    def get_number(cls):
        current_number=cls.query.first()
        if not current_number:
            db.session.add(cls(last_number="FS-F00001"))
            return "FS-F00001"
        else:
            last_number=current_number.last_number
            number=int(last_number[4:])
            number+=1
            number=str(number)
            delta=5-len(number)
            last= "FS-F{}{}".format("0"*delta,number)
            current_number.last_number=last
            return last

class NumCommande(db.Model):
    __tablename__="_num_commande"
    id=db.Column(db.Integer, primary_key=True)
    last_number=db.Column(db.String(50))
    @classmethod
    def get_number(cls):
        current_number=cls.query.first()
        if not current_number:
            db.session.add(cls(last_number="FS-BC0001"))
            return "FS-BC0001"
        else:
            last_number=current_number.last_number
            number=int(last_number[5:])
            number+=1
            number=str(number)
            delta=4-len(number)
            last= "FS-BC{}{}".format("0"*delta,number)
            current_number.last_number=last
            return last

class NumRecu(db.Model):
    __tablename__="_num_recu"
    id=db.Column(db.Integer, primary_key=True)
    last_number=db.Column(db.String(50)) # FS-R00001

    @classmethod
    def get_number(cls):
        current_number=cls.query.first()
        if not current_number:
            db.session.add(cls(last_number="FS-R00001"))
            return "FS-R00001"
        else:
            last_number=current_number.last_number
            number=int(last_number[4:])
            number+=1
            number=str(number)
            delta=5-len(number)
            last= "FS-R{}{}".format("0"*delta,number)
            current_number.last_number=last
            return last


class NumDepense(db.Model):
    __tablename__="_num_depense"
    id=db.Column(db.Integer, primary_key=True)
    last_number=db.Column(db.String(50)) # FS-R00001

    @classmethod
    def get_number(cls):
        current_number=cls.query.first()
        if not current_number:
            db.session.add(cls(last_number="FS-D00001"))
            return "FS-D00001"
        else:
            last_number=current_number.last_number
            number=int(last_number[4:])
            number+=1
            number=str(number)
            delta=5-len(number)
            last= "FS-D{}{}".format("0"*delta,number)
            current_number.last_number=last
            return last


class NumClient(db.Model):
    __tablename__="_num_client"
    id=db.Column(db.Integer, primary_key=True)
    last_number=db.Column(db.String(50))
    @classmethod
    def get_number(cls):
        current_number=cls.query.first()
        if not current_number:
            db.session.add(cls(last_number="FS-C001"))
            return "FS-C001"
        else:
            last_number=current_number.last_number
            number=int(last_number[4:])
            number+=1
            number=str(number)
            delta=3-len(number)
            last= "FS-C{}{}".format("0"*delta,number)
            current_number.last_number=last
            return last
